/**
 * \file caam_session.c
 *
 * \brief Architecture specific implementation of session management functions
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>

#include <private/sdc_arch.h>
#include <linux/caam_ee.h>
#include "caam.h"

#include <sdc/arch/errors.h>
#include <sdc/arch/common.h>
#include <sdc/arch/perm.h>

sdc_error_t sdc_arch_session_init(sdc_session_t *session)
{
    int fd;

    /* Open the caam device */
    fd = open(CAAM_DEVICE, O_RDWR, 0);
    if (fd < 0) {
        return SDC_IMX6_OPEN_CAAMEE_DEVICENODE_FAILED;
    }

    session->arch.fd = fd;
    session->arch.request = 0;
    memset(session->arch.iv, 0, CAAM_EE_MAX_ENC_BLOCK_ALIGN);
    session->arch.iv_len = 0;
    session->arch.buf_idx = 0;

    return SDC_OK;
}

sdc_error_t  sdc_arch_session_deinit(sdc_session_t *session)
{
    /* Close the device */
    close(session->arch.fd);

    return SDC_OK;
}

sdc_error_t sdc_arch_session_load_storage_key(sdc_session_t *session, sdc_key_id_t kid)
{
    struct caam_ee_session_keystore_key_data data;
    int res;

    /* Set up the ioctl data structure */
    data.key_id = (caam_key_id) kid;

    /* Call the kernel's CAAM_SET_SESSION_KEYSTORE_KEY */
    res = ioctl(session->arch.fd, CAAM_SET_SESSION_KEYSTORE_KEY, &data);

    if (res != 0) {
        return error_from_ioctl_errno(errno, CAAM_SET_SESSION_KEYSTORE_KEY, 0);
    }

    return SDC_OK;
}

sdc_error_t sdc_arch_session_load_builtin_key(sdc_session_t *session,
                                              const sdc_key_common_t *key_config,
                                              const sdc_key_modifier_t *key_modifier)
{
    struct caam_ee_session_builtin_key_data data;
    int res;
    sdc_error_t rtnval;
    size_t key_bits;

    if (key_config->fmt != SDC_KEY_FMT_SIMPLE_SYM)
        return SDC_NOT_SUPPORTED;

    rtnval = sdc_key_len_get_bits(key_config->len, &key_bits);
    if (rtnval != SDC_OK)
        return rtnval;

    if ((key_modifier->len > 0) && (!key_modifier->data))
        return SDC_INVALID_PARAMETER;

    /* Set up the ioctl data structure */
    data.spec.key_flags = CAAM_KEY_FLAG_STOR_BLACK |
                          CAAM_KEY_FLAG_TYPE_BUILTIN |
                          CAAM_KEY_FLAG_SUBTYPE_DERIVED |
                          CAAM_KEY_FLAG_SECRET_BYTES_PLAIN |
                          CAAM_KEY_FLAG_AUTH_HIGH |
                          CAAM_KEY_FLAG_CONFIDENT_HIGH |
                          CAAM_IMX6_USER_FLAGS__BUILTIN_KEY;

    data.spec.key_bits = key_bits;

    /* permissions handling */
    data.spec.uid = sdc_uid_to_caam_uid(key_config->perms->uid);
    data.spec.gid = sdc_gid_to_caam_gid(key_config->perms->gid);

    rtnval = sdc_permissions_to_caam_key_perms(key_config->perms, &data.spec.perms);
    if (rtnval)
        return rtnval;

    data.spec.modifier = NULL;
    data.spec.modifier_bytes = 0;
    if (key_modifier->len > 0) {
        data.spec.modifier = (uint8_t *)key_modifier->data;
        data.spec.modifier_bytes = (uint32_t)key_modifier->len;

        if (key_modifier->is_secret)
            data.spec.key_flags |= CAAM_KEY_FLAG_MODIFIER_PRIVATE;
        else
            data.spec.key_flags |= CAAM_KEY_FLAG_MODIFIER_PUBLIC;
    } else {
        data.spec.key_flags |= CAAM_KEY_FLAG_MODIFIER_NON;
    }

    /* Call the kernel's CAAM_SET_SESSION_BUILTIN_KEY */
    res = ioctl(session->arch.fd, CAAM_SET_SESSION_BUILTIN_KEY, &data);

    if (res != 0) {
        return error_from_ioctl_errno(errno, CAAM_SET_SESSION_BUILTIN_KEY, 0);
    }

    return SDC_OK;
}

sdc_error_t sdc_arch_session_unset_key(sdc_session_t *session)
{
    int res;

    /* Call the kernel's CAAM_UNSET_SESSION_KEY */
    res = ioctl(session->arch.fd, CAAM_UNSET_SESSION_KEY, NULL);
    if (res != 0) {
        return error_from_ioctl_errno(errno, CAAM_UNSET_SESSION_KEY, 0);
    }

    return SDC_OK;
}

static sdc_error_t convert_caam_key_info_to_sdc_key_info(const struct caam_key_info *caam_key_info,
                                                         sdc_key_info_t *key_info)
{
    sdc_error_t rtnval = SDC_OK;
    sdc_error_t tmprtnval;

    /* early abort in case of unset session key */
    if ((caam_key_info->key_flags & CAAM_KEY_FLAG_TYPE_MASK) == CAAM_KEY_FLAG_TYPE_INVALID)
        return SDC_KEY_UNSET;

    switch (caam_key_info->key_flags & CAAM_KEY_FLAG_SECRET_MASK) {
    case CAAM_KEY_FLAG_SECRET_BYTES_PLAIN:
        key_info->key_fmt = SDC_KEY_FMT_SIMPLE_SYM;
        break;
    case CAAM_KEY_FLAG_SECRET_RSA_BER_PRI:
        key_info->key_fmt = SDC_KEY_FMT_RSA_PRIVATE;
        break;
    case CAAM_KEY_FLAG_SECRET_RSA_BER_PUB:
        key_info->key_fmt = SDC_KEY_FMT_RSA_PUBLIC;
        break;
    default:
        rtnval = SDC_KEY_INFO_INVALID;
    }

    switch (caam_key_info->key_flags & CAAM_KEY_FLAG_TYPE_MASK) {
    case CAAM_KEY_FLAG_TYPE_KEYSTORE:
        key_info->kid = caam_key_info->key_id;

        switch (caam_key_info->key_flags & CAAM_KEY_FLAG_USER_MASK) {

        case CAAM_IMX6_USER_FLAGS__PRODUCT_KEYSTORE_KEY:
            if ((caam_key_info->key_flags & CAAM_KEY_FLAG_SUBTYPE_MASK) !=
                CAAM_KEY_FLAG_SUBTYPE_PLAIN) {
                rtnval = SDC_KEY_INFO_INVALID;
            }
            key_info->key_type = SDC_KEY_TYPE_KEYSTORE_PRODUCT;
            break;

        case CAAM_IMX6_USER_FLAGS__NORMAL_KEYSTORE_KEY:
            switch (caam_key_info->key_flags & CAAM_KEY_FLAG_SUBTYPE_MASK) {

            case CAAM_KEY_FLAG_SUBTYPE_PLAIN:
                /* also used for old keystore keys */
                key_info->key_type = SDC_KEY_TYPE_KEYSTORE_PLAIN;
                break;

            case CAAM_KEY_FLAG_SUBTYPE_RANDOM:
                key_info->key_type = SDC_KEY_TYPE_KEYSTORE_RANDOM;
                break;

            case CAAM_KEY_FLAG_SUBTYPE_IMPORTED:
                key_info->key_type = SDC_KEY_TYPE_KEYSTORE_IMPORTED;
                break;

            default:
                rtnval = SDC_KEY_INFO_INVALID;
            }
            break;

        default:
            rtnval = SDC_KEY_INFO_INVALID;
        }
        break;
    case CAAM_KEY_FLAG_TYPE_BUILTIN:
        if ((caam_key_info->key_flags & CAAM_KEY_FLAG_USER_MASK) !=
            CAAM_IMX6_USER_FLAGS__BUILTIN_KEY) {
            rtnval = SDC_KEY_INFO_INVALID;
        }
        if ((caam_key_info->key_flags & CAAM_KEY_FLAG_SUBTYPE_MASK) ==
            CAAM_KEY_FLAG_SUBTYPE_DERIVED) {
            key_info->key_type = SDC_KEY_TYPE_BUILT_IN;
        } else
            rtnval = SDC_KEY_INFO_INVALID;
        break;
    default:
        rtnval = SDC_KEY_INFO_INVALID;
    }

    key_info->key_len_bits = caam_key_info->key_bits;
    key_info->key_len_bytes = caam_key_info->key_bytes;

    /* permission handling */
    key_info->perms.uid = caam_uid_to_sdc_uid(caam_key_info->uid);
    key_info->perms.gid = caam_gid_to_sdc_gid(caam_key_info->gid);
    tmprtnval = caam_key_perms_to_sdc_permissions(&caam_key_info->perms, &key_info->perms);
    if (!rtnval)
        rtnval = tmprtnval;

    key_info->key_modifier.is_secret = false;
    key_info->key_modifier.data = NULL;
    key_info->key_modifier.len = 0;
    switch (caam_key_info->key_flags & CAAM_KEY_FLAG_MODIFIER_MASK) {
    case CAAM_KEY_FLAG_MODIFIER_PRIVATE:
        key_info->key_modifier.is_secret = true;
    /* FALLTHROUGH */
    case CAAM_KEY_FLAG_MODIFIER_NON:
        break;
    case CAAM_KEY_FLAG_MODIFIER_PUBLIC:
        if ((caam_key_info->modifier) &&
            (caam_key_info->modifier_bytes > 0)) {
            key_info->key_modifier.data = caam_key_info->modifier;
            key_info->key_modifier.len = caam_key_info->modifier_bytes;
        } else
            rtnval = SDC_KEY_INFO_INVALID;
        break;
    default:
        rtnval = SDC_KEY_INFO_INVALID;
    }

    switch (caam_key_info->key_flags & CAAM_KEY_FLAG_AUTH_MASK) {
    case CAAM_KEY_FLAG_AUTH_UNKNOWN:
        key_info->authenticity = SDC_KEY_AUTH_UNKNOWN;
        break;
    case CAAM_KEY_FLAG_AUTH_LOW:
        key_info->authenticity = SDC_KEY_AUTH_LOW;
        break;
    case CAAM_KEY_FLAG_AUTH_MEDIUM:
        key_info->authenticity = SDC_KEY_AUTH_MEDIUM;
        break;
    case CAAM_KEY_FLAG_AUTH_HIGH:
        key_info->authenticity = SDC_KEY_AUTH_HIGH;
        break;
    default:
        rtnval = SDC_KEY_INFO_INVALID;
    }

    switch (caam_key_info->key_flags & CAAM_KEY_FLAG_CONFIDENT_MASK) {
    case CAAM_KEY_FLAG_CONFIDENT_UNKNOWN:
        key_info->confidentiality = SDC_KEY_CONFIDENT_UNKNOWN;
        break;
    case CAAM_KEY_FLAG_CONFIDENT_LOW:
        key_info->confidentiality = SDC_KEY_CONFIDENT_LOW;
        break;
    case CAAM_KEY_FLAG_CONFIDENT_MEDIUM:
        key_info->confidentiality = SDC_KEY_CONFIDENT_MEDIUM;
        break;
    case CAAM_KEY_FLAG_CONFIDENT_HIGH:
        key_info->confidentiality = SDC_KEY_CONFIDENT_HIGH;
        break;
    default:
        rtnval = SDC_KEY_INFO_INVALID;
    }

    return rtnval;
}

/*
 * returns SDC_KEY_UNSET if key is invalid
 * returns SDC_OK if key is set
 * other errors in case getting key info failed
 */
// FIXME switch to key info after adding caching
sdc_error_t caam_check_session_key_set(sdc_session_t *session)
{
    sdc_error_t rtnval = SDC_OK;
    struct caam_key_info data;
    int res;

    memset(&data, 0, sizeof(struct caam_key_info));

    /* Call the kernel's CAAM_SET_SESSION_BUILTIN_KEY */
    res = ioctl(session->arch.fd, CAAM_GET_SESSION_KEY_INFO, &data);

    if (res != 0)
        rtnval = error_from_ioctl_errno(errno, CAAM_GET_SESSION_KEY_INFO, 0);

    if ((rtnval == SDC_OK) && ((data.key_flags & CAAM_KEY_FLAG_TYPE_MASK) == CAAM_KEY_FLAG_TYPE_INVALID))
        rtnval = SDC_KEY_UNSET;

    return rtnval;
}

// FIXME add caching of session key info
sdc_error_t sdc_arch_session_key_info_get(sdc_session_t *session,
                                          sdc_key_info_t *key_info, bool with_modifier)
{
    sdc_error_t rtnval = SDC_OK;
    struct caam_key_info data;
    int res;
    uint32_t modifier_len;

    memset(&data, 0, sizeof(struct caam_key_info));

    /* Call the kernel's CAAM_SET_SESSION_BUILTIN_KEY */
    res = ioctl(session->arch.fd, CAAM_GET_SESSION_KEY_INFO, &data);

    if ((with_modifier) && (res == 0) && (data.modifier_bytes > 0)) {
        modifier_len = data.modifier_bytes;

        /* prepare read again */
        memset(&data, 0, sizeof(struct caam_key_info));

        /* we now have the length of the modifier and can allocate memory */
        data.modifier = malloc(modifier_len);
        if (!data.modifier)
            return SDC_NO_MEM;

        data.modifier_bytes = modifier_len;

        /* Call the kernel's CAAM_SET_SESSION_BUILTIN_KEY */
        res = ioctl(session->arch.fd, CAAM_GET_SESSION_KEY_INFO, &data);
    }

    if (res != 0)
        rtnval = error_from_ioctl_errno(errno, CAAM_GET_SESSION_KEY_INFO, 0);

    if (rtnval == SDC_OK)
        rtnval = convert_caam_key_info_to_sdc_key_info(&data, key_info);

    if (rtnval != SDC_OK)
        free(data.modifier);

    return rtnval;
}
